//-----------------------------------------------------------------------------
//--
//-- module:    uart_baud_gen
//-- project:   wave_gen
//-- company:   Xilinx, Inc.
//-- author:    WK, AW
//-- 
//-- comment:
//--   Generates a 16x Baud enable. This signal is generated 16 times per  bit
//--   at the correct baud rate as determined by the parameters for the  system
//--   clock frequency and the Baud  rate
//--   
//-- 1) Divider must be at least 2 (thus CLOCK_RATE must be at least 32x
//--   BAUD_RATE)
//-- 
//-- known issues:
//-- status           id     found     description                      by fixed date  by    comment
//-- 
//-- version history:
//--   version    date    author     description
//--    11.1-001 20 APR 2009 WK       First version for 11.1          
//-- 
//-- ---------------------------------------------------------------------------
//-- 
//-- disclaimer:
//--   Disclaimer: LIMITED WARRANTY AND DISCLAMER. These designs  are
//--   provided to you as is . Xilinx and its licensors make, and  you
//--   receive no warranties or conditions, express,  implied,
//--   statutory or otherwise, and Xilinx specifically disclaims  any
//--   implied warranties of merchantability, non-infringement,  or
//--   fitness for a particular purpose. Xilinx does not warrant  that
//--   the functions contained in these designs will meet  your
//--   requirements, or that the operation of these designs will  be
//--   uninterrupted or error free, or that defects in the  Designs
//--   will be corrected. Furthermore, Xilinx does not warrant  or
//--   make any representations regarding use or the results of  the
//--   use of the designs in terms of correctness,  accuracy,
//--   reliability, or  otherwise.
//--   
//-- LIMITATION OF LIABILITY. In no event will Xilinx or  its
//--   licensors be liable for any loss of data, lost profits,  cost
//--   or procurement of substitute goods or services, or for  any
//--   special, incidental, consequential, or indirect  damages
//--   arising from the use or operation of the designs  or
//--   accompanying documentation, however caused and on any  theory
//--   of liability. This limitation will apply even if  Xilinx
//--   has been advised of the possibility of such damage.  This
//--   limitation shall apply not-withstanding the failure of  the
//--   essential purpose of any limited remedies  herein.
//--   
//-- Copyright  2002, 2008, 2009 Xilinx,  Inc.
//--   All rights reserved
//-- 
//-- -----------------------------------------------------------------------------

module uart_baud_gen #(
    parameter CLOCK_RATE = 100000000,                                       // clock rate
    parameter BAUD_RATE  = 115200                                          // desired baud rate
    )  (
    input  rst,                                                         // external reset in
    input  clk,                                                         // clock 
    output reg baud_x16_en                                                  // 16 times the baud rate
    );

parameter OVERSAMPLE_RATE = BAUD_RATE * 16;
parameter OVERSAMPLE_VALUE = (CLOCK_RATE+OVERSAMPLE_RATE/2)/OVERSAMPLE_RATE - 1; // one enable produced every this many counts

//clk_divider: process (clk)

reg [5:0] internal_count = 6'b0;                                        // internal counter

always @( posedge clk ) begin                                           // synchronous process
    if ( rst == 1'b1 ) begin                                            // if reset is active
        internal_count <= OVERSAMPLE_VALUE;                             // reset the count in preparation for count-down
        baud_x16_en    <= 1'b0;                                         // drive the external enable inactive
    end else begin                                                          // every 16xbaud interval, generate a one-clock enable pulse
        baud_x16_en <= 1'b0;                                            // hold the enable inactive            
        if ( internal_count == 0 ) begin                                // at terminal count?
            baud_x16_en    <= 1'b1;                                     // generate the active high enable
            internal_count <= OVERSAMPLE_VALUE;                         // reset the count
        end else begin    
            internal_count <= internal_count - 1;                       // decrement the counter   
        end                                                             // end of count reached
    end                                                                 // end of non-reset activities
end                                                                     // end of synchronous activities

//end process clk_divider;

endmodule

